<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly

abstract class Jet_Woo_Product_Gallery_Base extends Widget_Base {

	public $__context = 'render';

	protected function _register_controls() {

		$this->register_base_general_controls();

	  $this->register_product_gallery_controls();

	  $this->register_base_gallery_controls();

		$css_scheme = apply_filters(
			'jet-woo-product-gallery/base/css-scheme',
			array(
				'wrapper'                 => '.jet-woo-product-video',
				'overlay'                 => '.jet-woo-product-video__overlay',
				'video_popup_wrapper'     => '.jet-woo-product-video__popup-wrapper',
				'video_popup_overlay'     => '.jet-woo-product-video__popup-overlay',
				'video-popup-button'      => '.jet-woo-product-video__popup-button',
				'video-play-overlay'      => '.jet-woo-product-video__overlay',
				'video-play-button'       => '.jet-woo-product-video__play-button',
				'video-play-button-image' => '.jet-woo-product-video__play-button-image',
				'popup-button'            => '.jet-woo-product-video__popup-button',
				'popup-button-icon'       => '.jet-woo-product-video__popup-button-icon',
				'popup-button-image'      => '.jet-woo-product-video__popup-button-image',
				'photoswipe-trigger'      => '.jet-woo-product-gallery .jet-woo-product-gallery__trigger',
			)
		);

		$this->register_base_video_controls( $css_scheme );

		$this->register_base_photoswipe_trigger_controls_style( $css_scheme );

		$this->register_base_video_popup_button_controls_style( $css_scheme );

		$this->register_base_video_play_button_controls_style( $css_scheme );

	}

	protected function register_base_general_controls() {

		$this->start_controls_section(
			'section_general_content',
			array(
				'label'      => esc_html__( 'General', 'jet-woo-product-gallery' ),
				'tab'        => Controls_Manager::TAB_CONTENT,
				'show_label' => false,
			)
		);

		$this->add_control(
			'product_id',
			array(
				'label' => esc_html__( 'Product id', 'jet-woo-product-gallery' ),
				'type'  => Controls_Manager::NUMBER,
			)
		);

		$this->add_control(
			'enable_gallery',
			array(
				'label'        => esc_html__( 'Enable Gallery', 'jet-woo-product-gallery' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'jet-woo-product-gallery' ),
				'label_off'    => esc_html__( 'No', 'jet-woo-product-gallery' ),
				'return_value' => 'yes',
				'default'      => '',
			)
		);

		$this->add_control(
			'enable_zoom',
			array(
				'label'        => esc_html__( 'Enable Zoom', 'jet-woo-product-gallery' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'jet-woo-product-gallery' ),
				'label_off'    => esc_html__( 'No', 'jet-woo-product-gallery' ),
				'return_value' => 'yes',
				'default'      => '',
			)
		);

		$this->end_controls_section();

	}

	protected function register_base_gallery_controls() {

		$this->start_controls_section(
			'section_gallery_style',
			array(
				'label'      => esc_html__( 'Gallery', 'jet-woo-product-gallery' ),
				'tab'        => Controls_Manager::TAB_CONTENT,
				'show_label' => false,
				'condition'  => array(
					'enable_gallery' => array( 'yes' )
				)
			)
		);

		$this->add_control(
			'gallery_button_icon',
			array(
				'label'       => esc_html__( 'Button Icon', 'jet-woo-product-gallery' ),
				'type'        => Controls_Manager::ICON,
				'label_block' => true,
				'default'     => 'fa fa-search',
			)
		);

		$this->add_control(
			'gallery_show_caption',
			array(
				'label'        => esc_html__( 'Show Caption', 'jet-woo-product-gallery' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'jet-woo-product-gallery' ),
				'label_off'    => esc_html__( 'No', 'jet-woo-product-gallery' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			)
		);

		$this->add_control(
			'gallery_controls_heading',
			array(
				'label'     => esc_html__( 'Controls', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'gallery_show_fullscreen',
			array(
				'label'        => esc_html__( 'Show Full Screen', 'jet-woo-product-gallery' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'jet-woo-product-gallery' ),
				'label_off'    => esc_html__( 'No', 'jet-woo-product-gallery' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			)
		);

		$this->add_control(
			'gallery_show_zoom',
			array(
				'label'        => esc_html__( 'Show Zoom', 'jet-woo-product-gallery' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'jet-woo-product-gallery' ),
				'label_off'    => esc_html__( 'No', 'jet-woo-product-gallery' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			)
		);

		$this->add_control(
			'gallery_show_share',
			array(
				'label'        => esc_html__( 'Show Share', 'jet-woo-product-gallery' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'jet-woo-product-gallery' ),
				'label_off'    => esc_html__( 'No', 'jet-woo-product-gallery' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			)
		);

		$this->add_control(
			'gallery_show_counter',
			array(
				'label'        => esc_html__( 'Show Counter', 'jet-woo-product-gallery' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'jet-woo-product-gallery' ),
				'label_off'    => esc_html__( 'No', 'jet-woo-product-gallery' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			)
		);

		$this->add_control(
			'gallery_show_arrows',
			array(
				'label'        => esc_html__( 'Show Arrows', 'jet-woo-product-gallery' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'jet-woo-product-gallery' ),
				'label_off'    => esc_html__( 'No', 'jet-woo-product-gallery' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			)
		);

		$this->end_controls_section();

	}

	protected function register_base_video_controls( $css_scheme ) {

		$this->start_controls_section(
			'section_video',
			array(
				'label' => esc_html__( 'Video', 'jet-woo-product-gallery' ),
			)
		);

		$this->add_control(
			'video_display_in',
			array(
				'label'   => esc_html__( 'Display Video In', 'jet-woo-product-gallery' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'content',
				'options' => array(
					'content' => esc_html__( 'Content', 'jet-woo-product-gallery' ),
					'popup'   => esc_html__( 'Popup', 'jet-woo-product-gallery' ),
				),
			)
		);

		$this->add_control(
			'aspect_ratio',
			array(
				'label'       => esc_html__( 'Aspect Ratio', 'jet-woo-product-gallery' ),
				'description' => esc_html__( 'Worked just with youtube and vimeo video types', 'jet-woo-product-gallery' ),
				'type'        => Controls_Manager::SELECT,
				'default'     => '16-9',
				'options'     => array(
					'16-9' => '16:9',
					'21-9' => '21:9',
					'4-3'  => '4:3',
					'3-2'  => '3:2',
					'1-1'  => '1:1',
				),
			)
		);

		$this->add_control(
			'video_options_heading',
			array(
				'label'     => esc_html__( 'Options', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'autoplay',
			array(
				'label'   => esc_html__( 'Autoplay', 'jet-woo-product-gallery' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => '',
			)
		);

		$this->add_control(
			'loop',
			array(
				'label'   => esc_html__( 'Loop', 'jet-woo-product-gallery' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => '',
			)
		);

		$this->register_product_video_in_content_controls( $css_scheme );

		$this->register_product_video_in_popup_controls( $css_scheme );

		$this->end_controls_section();

	}

	protected function register_product_video_in_content_controls( $css_scheme ) {

		$this->add_control(
			'video_overlay_heading',
			array(
				'label'     => esc_html__( 'Overlay', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => array(
					'video_display_in' => 'content'
				)
			)
		);

		$this->add_control(
			'overlay_color',
			array(
				'label'     => esc_html__( 'Overlay Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['overlay'] . ':before' => 'background-color: {{VALUE}};',
				),
				'condition' => array(
					'video_display_in' => 'content'
				)
			)
		);

		$this->add_control(
			'video_play_button_heading',
			array(
				'label'     => esc_html__( 'Play Button', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => array(
					'video_display_in' => 'content'
				)
			)
		);

		$this->add_control(
			'show_play_button',
			array(
				'label'     => esc_html__( 'Show Play Button', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => array(
					'video_display_in' => 'content'
				)
			)
		);

		$this->add_control(
			'play_button_type',
			array(
				'label'     => esc_html__( 'Play Button Type', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::CHOOSE,
				'default'   => 'icon',
				'toggle'    => false,
				'options'   => array(
					'icon'  => array(
						'title' => esc_html__( 'Icon', 'jet-woo-product-gallery' ),
						'icon'  => 'fa fa-play',
					),
					'image' => array(
						'title' => esc_html__( 'Image', 'jet-woo-product-gallery' ),
						'icon'  => 'fa fa-picture-o',
					)
				),
				'condition' => array(
					'video_display_in' => 'content',
					'show_play_button' => 'yes',
				),
			)
		);

		$this->add_control(
			'play_button_icon',
			array(
				'label'     => esc_html__( 'Icon', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::ICON,
				'default'   => 'fa fa-play',
				'condition' => array(
					'video_display_in' => 'content',
					'show_play_button' => 'yes',
					'play_button_type' => 'icon',
				),
			)
		);

		$this->add_control(
			'play_button_image',
			array(
				'label'     => esc_html__( 'Image', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::MEDIA,
				'condition' => array(
					'video_display_in' => 'content',
					'show_play_button' => 'yes',
					'play_button_type' => 'image',
				),
			)
		);

	}

	protected function register_product_video_in_popup_controls( $css_scheme ) {

		$this->add_control(
			'popup_video_overlay_heading',
			array(
				'label'     => esc_html__( 'Popup Overlay', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => array(
					'video_display_in' => 'popup'
				)
			)
		);

		$this->add_control(
			'popup_overlay_color',
			array(
				'label'     => esc_html__( 'Overlay Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['video_popup_overlay'] => 'background-color: {{VALUE}};',
				),
				'condition' => array(
					'video_display_in' => 'popup'
				)
			)
		);

		$this->add_control(
			'video_popup_button_heading',
			array(
				'label'     => esc_html__( 'Popup Button', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => array(
					'video_display_in' => 'popup'
				)
			)
		);

		$this->add_control(
			'popup-button-icon',
			array(
				'label'     => esc_html__( 'Icon', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::ICON,
				'default'   => 'fa fa-video-camera',
				'condition' => array(
					'video_display_in' => 'popup',
				),
			)
		);

	}

	/**
	 * Style controls for base elements.
	 */
	protected function register_base_photoswipe_trigger_controls_style( $css_scheme ) {
		$this->start_controls_section(
			'section_photoswipe_trigger_style',
			array(
				'label'      => esc_html__( 'Photoswipe Trigger', 'jet-woo-product-gallery' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			)
		);

		$this->add_control(
			'photoswipe_trigger_show_on_hover',
			array(
				'label'        => esc_html__( 'Show On Hover', 'jet-woo-product-gallery' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'jet-woo-product-gallery' ),
				'label_off'    => esc_html__( 'No', 'jet-woo-product-gallery' ),
				'return_value' => 'show-on-hover',
				'default'      => '',
				'prefix_class' => 'jet-woo-product-gallery__trigger--',
			)
		);

		$this->add_control(
			'photoswipe_trigger_position',
			array(
				'label'        => esc_html__( 'Position', 'jet-woo-product-gallery' ),
				'type'         => Controls_Manager::SELECT,
				'default'      => 'top-right',
				'options'      => array(
					'top-right'    => esc_html__( 'Top Right', 'jet-woo-product-gallery' ),
					'bottom-right' => esc_html__( 'Bottom Right', 'jet-woo-product-gallery' ),
					'bottom-left'  => esc_html__( 'Bottom Left', 'jet-woo-product-gallery' ),
					'top-left'     => esc_html__( 'Top Left', 'jet-woo-product-gallery' ),
					'center'       => esc_html__( 'Center Center', 'jet-woo-product-gallery' ),
				),
				'prefix_class' => 'jet-woo-product-gallery__trigger--',
			)
		);

		$this->add_responsive_control(
			'photoswipe_trigger_size',
			array(
				'label'      => esc_html__( 'Size', 'jet-woo-product-gallery' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array(
					'px',
				),
				'range'      => array(
					'px' => array(
						'min' => 20,
						'max' => 200,
					),
				),
				'default'    => array(
					'size' => 30,
					'unit' => 'px',
				),
				'selectors'  => array(
					'{{WRAPPER}} ' . $css_scheme['photoswipe-trigger'] => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'photoswipe_trigger_icon_size',
			array(
				'label'      => esc_html__( 'Icon Size', 'jet-woo-product-gallery' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array(
					'px',
				),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 50,
					),
				),
				'default'    => array(
					'size' => 18,
					'unit' => 'px',
				),
				'selectors'  => array(
					'{{WRAPPER}} ' . $css_scheme['photoswipe-trigger'] . ' .jet-woo-product-gallery__trigger-icon' => 'font-size: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->start_controls_tabs( 'photoswipe_trigger_style_tabs' );

		$this->start_controls_tab(
			'photoswipe_trigger_normal_styles',
			array(
				'label' => esc_html__( 'Normal', 'jet-woo-product-gallery' ),
			)
		);

		$this->add_control(
			'photoswipe_trigger_normal_color',
			array(
				'label'     => esc_html__( 'Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['photoswipe-trigger'] . ' .jet-woo-product-gallery__trigger-icon' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'photoswipe_trigger_normal_background_color',
			array(
				'label'     => esc_html__( ' Background Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['photoswipe-trigger'] => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'photoswipe_trigger_hover_styles',
			array(
				'label' => esc_html__( 'Hover', 'jet-woo-product-gallery' ),
			)
		);

		$this->add_control(
			'photoswipe_trigger_hover_color',
			array(
				'label'     => esc_html__( 'Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['photoswipe-trigger'] . ':hover' . ' .jet-woo-product-gallery__trigger-icon' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'photoswipe_trigger_hover_background_color',
			array(
				'label'     => esc_html__( 'Background Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['photoswipe-trigger'] . ':hover' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'photoswipe_trigger_hover_border_color',
			array(
				'label'     => esc_html__( 'Border Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['photoswipe-trigger'] . ':hover' => 'border-color: {{VALUE}}',
				),
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'        => 'photoswipe_trigger_border',
				'label'       => esc_html__( 'Border', 'jet-woo-product-gallery' ),
				'placeholder' => '1px',
				'default'     => '1px',
				'selector'    => '{{WRAPPER}} ' . $css_scheme['photoswipe-trigger'],
			)
		);

		$this->add_control(
			'photoswipe_trigger_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius', 'jet-woo-product-gallery' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} ' . $css_scheme['photoswipe-trigger'] => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}; overflow:hidden;',
				),
			)
		);

		$this->add_responsive_control(
			'photoswipe_trigger_margin',
			array(
				'label'      => esc_html__( 'Margin', 'jet-woo-product-gallery' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} ' . $css_scheme['photoswipe-trigger'] => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();
	}

	protected function register_base_video_popup_button_controls_style( $css_scheme ) {
		$this->start_controls_section(
			'video_popup_button_style',
			array(
				'label'      => esc_html__( 'Video Popup Button', 'jet-woo-product-gallery' ),
				'tab'        => Controls_Manager::TAB_STYLE,
				'show_label' => false,
			)
		);

		$this->add_responsive_control(
			'video_popup_button_icon_size',
			array(
				'label'      => esc_html__( 'Icon Size', 'jet-woo-product-gallery' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array(
					'px',
				),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 50,
					),
				),
				'default'    => array(
					'size' => 18,
					'unit' => 'px',
				),
				'selectors'  => array(
					'{{WRAPPER}} ' . $css_scheme['video-popup-button'] . ' .jet-woo-product-video__popup-button-icon' => 'font-size: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->start_controls_tabs( 'video_popup_button_style_tabs' );

		$this->start_controls_tab(
			'video_popup_button_normal_styles',
			array(
				'label' => esc_html__( 'Normal', 'jet-woo-product-gallery' ),
			)
		);

		$this->add_control(
			'video_popup_button_normal_color',
			array(
				'label'     => esc_html__( 'Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['video-popup-button'] . ' .jet-woo-product-video__popup-button-icon' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'video_popup_button_normal_background_color',
			array(
				'label'     => esc_html__( ' Background Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['video-popup-button'] => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'video_popup_button_hover_styles',
			array(
				'label' => esc_html__( 'Hover', 'jet-woo-product-gallery' ),
			)
		);

		$this->add_control(
			'video_popup_button_hover_color',
			array(
				'label'     => esc_html__( 'Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['video-popup-button'] . ':hover' . ' .jet-woo-product-video__popup-button-icon' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'video_popup_button_hover_background_color',
			array(
				'label'     => esc_html__( 'Background Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['video-popup-button'] . ':hover' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'video_popup_button_hover_border_color',
			array(
				'label'     => esc_html__( 'Border Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['video-popup-button'] . ':hover' => 'border-color: {{VALUE}}',
				),
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'        => 'video_popup_button_border',
				'label'       => esc_html__( 'Border', 'jet-woo-product-gallery' ),
				'placeholder' => '1px',
				'default'     => '1px',
				'selector'    => '{{WRAPPER}} ' . $css_scheme['video-popup-button'],
			)
		);

		$this->add_control(
			'video_popup_button_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius', 'jet-woo-product-gallery' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} ' . $css_scheme['video-popup-button'] => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}; overflow:hidden;',
				),
			)
		);

		$this->add_responsive_control(
			'video_popup_button_padding',
			array(
				'label'      => esc_html__( 'Padding', 'jet-woo-product-gallery' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} ' . $css_scheme['video-popup-button'] => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'video_popup_button_margin',
			array(
				'label'      => esc_html__( 'Margin', 'jet-woo-product-gallery' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} ' . $css_scheme['video-popup-button'] => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);
		
		$this->add_responsive_control(
			'video_popup_button_alignment',
			array(
				'label'   => esc_html__( 'Alignment', 'jet-woo-product-gallery' ),
				'type'    => Controls_Manager::CHOOSE,
				'options' => array(
					'left' => array(
						'title' => esc_html__( 'Left', 'jet-woo-product-gallery' ),
						'icon'  => 'fa fa-align-left',
					),
					'center' => array(
						'title' => esc_html__( 'Center', 'jet-woo-product-gallery' ),
						'icon'  => 'fa fa-align-center',
					),
					'right' => array(
						'title' => esc_html__( 'Right', 'jet-woo-product-gallery' ),
						'icon'  => 'fa fa-align-right',
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} ' . $css_scheme['video_popup_wrapper']  => 'text-align: {{VALUE}};',
				),
			)
		);

		$this->end_controls_section();
	}

	protected function register_base_video_play_button_controls_style( $css_scheme ) {
		$this->start_controls_section(
			'section_video_play_button_style',
			array(
				'label' => esc_html__( 'Play Button', 'jet-woo-product-gallery' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_responsive_control(
			'video_play_button_size',
			array(
				'label'     => esc_html__( 'Icon/Image Size', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 10,
						'max' => 300,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['video-play-button'] => 'font-size: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'video_play_button_image_border_radius',
			array(
				'label'      => esc_html__( 'Image Border Radius', 'jet-woo-product-gallery' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} ' . $css_scheme['video-play-button-image'] => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
				'condition'  => array(
					'video_play_button_type' => 'image',
				),
			)
		);

		$this->start_controls_tabs( 'video_play_button_tabs' );

		$this->start_controls_tab( 'video_play_button_normal_tab',
			array(
				'label' => esc_html__( 'Normal', 'jet-woo-product-gallery' ),
			)
		);

		$this->add_control(
			'video_play_button_color',
			array(
				'label'     => esc_html__( 'Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['video-play-button'] => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'video_play_button__background_color',
			array(
				'label'     => esc_html__( 'Background Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['video-play-button'] => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'video_play_button_box_shadow',
				'selector' => '{{WRAPPER}} ' . $css_scheme['video-play-button'],
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab( 'video_play_button_hover_tab',
			array(
				'label' => esc_html__( 'Hover', 'jet-woo-product-gallery' ),
			)
		);

		$this->add_control(
			'video_play_button_color_hover',
			array(
				'label'     => esc_html__( 'Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['video-play-overlay'] . ':hover ' . $css_scheme['video-play-button'] => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'video_play_button_hover_background_color',
			array(
				'label'     => esc_html__( 'Background Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['video-play-overlay'] . ':hover ' . $css_scheme['video-play-button'] => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'video_play_button_border_color_hover',
			array(
				'label'     => esc_html__( 'Border Color', 'jet-woo-product-gallery' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} ' . $css_scheme['video-play-overlay'] . ':hover ' . $css_scheme['video-play-button'] => 'border-color: {{VALUE}};',
				),
				'condition' => array(
					'video_play_button_border_border!' => '',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'video_play_button_box_shadow_hover',
				'selector' => '{{WRAPPER}} ' . $css_scheme['video-play-overlay'] . ':hover ' . $css_scheme['video-play-button'],
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'video_play_button_padding',
			array(
				'label'      => esc_html__( 'Padding', 'jet-woo-product-gallery' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'separator'  => 'before',
				'selectors'  => array(
					'{{WRAPPER}} ' . $css_scheme['video-play-button'] => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'video_play_button_margin',
			array(
				'label'      => esc_html__( 'Margin', 'jet-woo-product-gallery' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} ' . $css_scheme['video-play-button'] => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'video_play_button_border',
				'selector' => '{{WRAPPER}} ' . $css_scheme['video-play-button'],
			)
		);

		$this->add_control(
			'video_play_button_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius', 'jet-woo-product-gallery' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} ' . $css_scheme['video-play-button'] => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Register widget style controls. Specific for each widget.
	 *
	 * @return void
	 */
	public function register_product_gallery_controls() {
	}

	/**
	 * Get globaly affected template
	 *
	 * @param  [type] $name [description]
	 *
	 * @return [type]       [description]
	 */
	public function __get_global_template( $name = null ) {

		$template = call_user_func( array( $this, sprintf( '__get_%s_template', $this->__context ) ) );

		if ( ! $template ) {
			$template = jet_woo_product_gallery()->get_template( $this->get_name() . '/global/' . $name . '.php' );
		}

		return $template;
	}

	/**
	 * Get front-end template
	 *
	 * @param  [type] $name [description]
	 *
	 * @return [type]       [description]
	 */
	public function __get_render_template( $name = null ) {
		return jet_woo_product_gallery()->get_template( $this->get_name() . '/render/' . $name . '.php' );
	}

	/**
	 * Get editor template
	 *
	 * @param  [type] $name [description]
	 *
	 * @return [type]       [description]
	 */
	public function __get_edit_template( $name = null ) {
		return jet_woo_product_gallery()->get_template( $this->get_name() . '/edit/' . $name . '.php' );
	}

	/**
	 * Get global looped template for settings
	 * Required only to process repeater settings.
	 *
	 * @param  string $name Base template name.
	 * @param  string $setting Repeater setting that provide data for template.
	 *
	 * @return void
	 */
	public function __get_global_looped_template( $name = null, $setting = null ) {

		$templates = array(
			'start' => $this->__get_global_template( $name . '-loop-start' ),
			'loop'  => $this->__get_global_template( $name . '-loop-item' ),
			'end'   => $this->__get_global_template( $name . '-loop-end' ),
		);

		call_user_func(
			array( $this, sprintf( '__get_%s_looped_template', $this->__context ) ), $templates, $setting
		);

	}

	/**
	 * Get render mode looped template
	 *
	 * @param  array $templates [description]
	 * @param  [type] $setting   [description]
	 *
	 * @return [type]            [description]
	 */
	public function __get_render_looped_template( $templates = array(), $setting = null ) {

		$loop = $this->get_settings( $setting );

		if ( empty( $loop ) ) {
			return;
		}

		if ( ! empty( $templates['start'] ) ) {
			include $templates['start'];
		}

		foreach ( $loop as $item ) {

			$this->__processed_item = $item;
			if ( ! empty( $templates['start'] ) ) {
				include $templates['loop'];
			}
			$this->__processed_index ++;
		}

		$this->__processed_item  = false;
		$this->__processed_index = 0;

		if ( ! empty( $templates['end'] ) ) {
			include $templates['end'];
		}

	}

	/**
	 * Get edit mode looped template
	 *
	 * @param  array $templates [description]
	 * @param  [type] $setting   [description]
	 *
	 * @return [type]            [description]
	 */
	public function __get_edit_looped_template( $templates = array(), $setting = null ) {
		?>
			<# if ( settings.<?php echo $setting; ?> ) { #>
		<?php
		if ( ! empty( $templates['start'] ) ) {
			include $templates['start'];
		}
		?>
			<# _.each( settings.<?php echo $setting; ?>, function( item ) { #>
		<?php
		if ( ! empty( $templates['loop'] ) ) {
			include $templates['loop'];
		}
		?>
			<# } ); #>
		<?php
		if ( ! empty( $templates['end'] ) ) {
			include $templates['end'];
		}
		?>
			<# } #>
		<?php
	}

	/**
	 * Get current looped item dependends from context.
	 *
	 * @param  string $key Key to get from processed item
	 *
	 * @return mixed
	 */
	public function __loop_item( $keys = array(), $format = '%s' ) {

		return call_user_func( array( $this, sprintf( '__%s_loop_item', $this->__context ) ), $keys, $format );

	}

	/**
	 * Loop edit item
	 *
	 * @param  [type]  $keys       [description]
	 * @param  string $format [description]
	 * @param  boolean $nested_key [description]
	 *
	 * @return [type]              [description]
	 */
	public function __edit_loop_item( $keys = array(), $format = '%s' ) {

		$settings = $keys[0];

		if ( isset( $keys[1] ) ) {
			$settings .= '.' . $keys[1];
		}

		ob_start();

		echo '<# if ( item.' . $settings . ' ) { #>';
		printf( $format, '{{{ item.' . $settings . ' }}}' );
		echo '<# } #>';

		return ob_get_clean();
	}

	/**
	 * Loop render item
	 *
	 * @param  string $format [description]
	 * @param  [type]  $key        [description]
	 * @param  boolean $nested_key [description]
	 *
	 * @return [type]              [description]
	 */
	public function __render_loop_item( $keys = array(), $format = '%s' ) {

		$item = $this->__processed_item;

		$key        = $keys[0];
		$nested_key = isset( $keys[1] ) ? $keys[1] : false;

		if ( empty( $item ) || ! isset( $item[ $key ] ) ) {
			return false;
		}

		if ( false === $nested_key || ! is_array( $item[ $key ] ) ) {
			$value = $item[ $key ];
		} else {
			$value = isset( $item[ $key ][ $nested_key ] ) ? $item[ $key ][ $nested_key ] : false;
		}

		if ( ! empty( $value ) ) {
			return sprintf( $format, $value );
		}

	}

	/**
	 * Include global template if any of passed settings is defined
	 *
	 * @param  [type] $name     [description]
	 * @param  [type] $settings [description]
	 *
	 * @return [type]           [description]
	 */
	public function __glob_inc_if( $name = null, $settings = array() ) {

		$template = $this->__get_global_template( $name );

		call_user_func( array( $this, sprintf( '__%s_inc_if', $this->__context ) ), $template, $settings );

	}

	/**
	 * Include render template if any of passed setting is not empty
	 *
	 * @param  [type] $file     [description]
	 * @param  [type] $settings [description]
	 *
	 * @return [type]           [description]
	 */
	public function __render_inc_if( $file = null, $settings = array() ) {

		foreach ( $settings as $setting ) {
			$val = $this->get_settings( $setting );

			if ( ! empty( $val ) ) {
				include $file;

				return;
			}

		}

	}

	/**
	 * Include render template if any of passed setting is not empty
	 *
	 * @param  [type] $file     [description]
	 * @param  [type] $settings [description]
	 *
	 * @return [type]           [description]
	 */
	public function __edit_inc_if( $file = null, $settings = array() ) {

		$condition = null;
		$sep       = null;

		foreach ( $settings as $setting ) {
			$condition .= $sep . 'settings.' . $setting;
			$sep       = ' || ';
		}

		?>

			<# if ( <?php echo $condition; ?> ) { #>

		<?php include $file; ?>

			<# } #>

		<?php
	}

	/**
	 * Open standard wrapper
	 *
	 * @return void
	 */
	public function __open_wrap() {
		printf(
			'<div class="elementor-%s jet-woo-product-gallery" data-gallery-settings=%s >',
			$this->get_name(),
			$this->__generate_gallery_setting_json()
		);
	}

	/**
	 * Close standard wrapper
	 *
	 * @return void
	 */
	public function __close_wrap() {
		echo '</div>';
	}

	/**
	 * Print HTML markup if passed setting not empty.
	 *
	 * @param  string $setting Passed setting.
	 * @param  string $format Required markup.
	 * @param  array $args Additional variables to pass into format string.
	 * @param  bool $echo Echo or return.
	 *
	 * @return string|void
	 */
	public function __html( $setting = null, $format = '%s' ) {

		call_user_func( array( $this, sprintf( '__%s_html', $this->__context ) ), $setting, $format );

	}

	/**
	 * Returns HTML markup if passed setting not empty.
	 *
	 * @param  string $setting Passed setting.
	 * @param  string $format Required markup.
	 * @param  array $args Additional variables to pass into format string.
	 * @param  bool $echo Echo or return.
	 *
	 * @return string|void
	 */
	public function __get_html( $setting = null, $format = '%s' ) {

		ob_start();
		$this->__html( $setting, $format );

		return ob_get_clean();

	}

	/**
	 * Print HTML template
	 *
	 * @param  [type] $setting [description]
	 * @param  [type] $format  [description]
	 *
	 * @return [type]          [description]
	 */
	public function __render_html( $setting = null, $format = '%s' ) {

		if ( is_array( $setting ) ) {
			$key     = $setting[1];
			$setting = $setting[0];
		}

		$val = $this->get_settings( $setting );

		if ( ! is_array( $val ) && '0' === $val ) {
			printf( $format, $val );
		}

		if ( is_array( $val ) && empty( $val[ $key ] ) ) {
			return '';
		}

		if ( ! is_array( $val ) && empty( $val ) ) {
			return '';
		}

		if ( is_array( $val ) ) {
			printf( $format, $val[ $key ] );
		} else {
			printf( $format, $val );
		}

	}

	/**
	 * Print underscore template
	 *
	 * @param  [type] $setting [description]
	 * @param  [type] $format  [description]
	 *
	 * @return [type]          [description]
	 */
	public function __edit_html( $setting = null, $format = '%s' ) {

		if ( is_array( $setting ) ) {
			$setting = $setting[0] . '.' . $setting[1];
		}

		echo '<# if ( settings.' . $setting . ' ) { #>';
		printf( $format, '{{{ settings.' . $setting . ' }}}' );
		echo '<# } #>';
	}

	/**
	 * Generate setting json
	 *
	 * @return string
	 */
	public function __generate_gallery_setting_json() {
		$module_settings = $this->get_settings();

		$settings = array(
			'enableGallery' => filter_var( $module_settings['enable_gallery'], FILTER_VALIDATE_BOOLEAN ),
			'enableZoom'    => filter_var( $module_settings['enable_zoom'], FILTER_VALIDATE_BOOLEAN ),
			'caption'       => filter_var( $module_settings['gallery_show_caption'], FILTER_VALIDATE_BOOLEAN ),
			'zoom'          => filter_var( $module_settings['gallery_show_zoom'], FILTER_VALIDATE_BOOLEAN ),
			'fullscreen'    => filter_var( $module_settings['gallery_show_fullscreen'], FILTER_VALIDATE_BOOLEAN ),
			'share'         => filter_var( $module_settings['gallery_show_share'], FILTER_VALIDATE_BOOLEAN ),
			'counter'       => filter_var( $module_settings['gallery_show_counter'], FILTER_VALIDATE_BOOLEAN ),
			'arrows'        => filter_var( $module_settings['gallery_show_arrows'], FILTER_VALIDATE_BOOLEAN ),
		);

		if ( $this->product_has_video() ) {
			$settings['hasVideo']      = true;
			$settings['videoType']     = jet_woo_gallery_video_integration()->get_video_type();
			$settings['videoIn']       = $module_settings['video_display_in'];
			$settings['videoAutoplay'] = filter_var( $module_settings['autoplay'], FILTER_VALIDATE_BOOLEAN );
			$settings['videoLoop']     = filter_var( $module_settings['loop'], FILTER_VALIDATE_BOOLEAN );
		}

		$settings = json_encode( $settings );

		return sprintf( '\'%1$s\'', $settings );
	}

	public function product_has_video() {
		$video_url = $this->__get_video_url();

		if ( empty( $video_url ) ) {
			return false;
		}

		return true;

	}

	/**
	 * Return url on iframe video placeholder
	 *
	 * @param $type
	 * @param $url
	 *
	 * @return string
	 */
	public function __get_video_iframe_thumbnail_url( $type, $url ) {
		$oembed = _wp_oembed_get_object();
		$data   = $oembed->get_data( $url );

		$thumb_url = $data->thumbnail_url;

		if ( 'youtube' === $type ) {
			$thumb_url = str_replace( '/hqdefault.', '/maxresdefault.', $thumb_url );
		}

		return esc_url( $thumb_url );
	}

	/**
	 * Check if video has custom placeholder
	 *
	 * @return bool
	 */
	public function __video_has_custom_placeholder() {
		return ! empty( jet_woo_gallery_video_integration()->get_video_custom_placeholder() );
	}

	/**
	 * Return url for video thumbnail
	 *
	 * @return string
	 */
	public function __get_video_thumbnail_url() {
		$thumb_url  = '';
		$video_url  = $this->__get_video_url();
		$video_type = jet_woo_gallery_video_integration()->get_video_type();

		if ( ! $this->product_has_video() ) {
			return '';
		}

		if ( $this->__video_has_custom_placeholder() ) {
			$video_placeholder_id  = jet_woo_gallery_video_integration()->get_video_custom_placeholder();
			$video_placeholder_src = wp_get_attachment_image_src( $video_placeholder_id, 'full' );
			$thumb_url             = $video_placeholder_src[0];
		} elseif ( in_array( $video_type, array( 'youtube', 'vimeo' ) ) ) {
			$thumb_url = $this->__get_video_iframe_thumbnail_url( $video_type, $video_url );
		}

		if ( empty( $thumb_url ) ) {
			return '';
		}

		return esc_url( $thumb_url );
	}

	/**
	 * Return generated video html
	 *
	 * @return bool|string
	 */
	public function __get_video_html() {
		$settings   = $this->get_settings();
		$video_url  = $this->__get_video_url();
		$video_type = jet_woo_gallery_video_integration()->get_video_type();
		$video_html = '';

		if ( ! $this->product_has_video() ) {
			return '';
		}

		if ( 'self_hosted' === $video_type ) {
			$self_hosted_params = $this->__get_self_hosted_video_params();

			$this->add_render_attribute( 'video_player', 'class', 'jet-woo-product-video-player' );
			$this->add_render_attribute( 'video_player', 'class', 'jet-woo-product-video-mejs-player' );
			$this->add_render_attribute( 'video_player', 'src', $video_url );
			$this->add_render_attribute( 'video_player', $self_hosted_params );

			if ( filter_var( $settings['show_play_button'], FILTER_VALIDATE_BOOLEAN ) ) {
				$this->add_render_attribute( 'video_player', 'class', 'jet-woo-product-video-custom-play-button' );
			}

			$video_html = '<video ' . $this->get_render_attribute_string( 'video_player' ) . '></video>';
		} else {
			$embed_params  = $this->__get_embed_video_params();
			$embed_options = array(
				'lazy_load' => false,
			);

			$embed_attr = array(
				'class' => 'jet-woo-product-video-iframe',
				'allow' => 'autoplay;encrypted-media',
			);

			$video_html = Embed::get_embed_html( $video_url, $embed_params, $embed_options, $embed_attr );

		}

		return $video_html;
	}

	/**
	 * Return parameters for self hosted video
	 *
	 * @return array
	 */
	public function __get_self_hosted_video_params() {
		$settings = $this->get_settings_for_display();

		$params = array();

		if ( 'content' === $settings['video_display_in'] ) {
			if ( filter_var( $settings['autoplay'], FILTER_VALIDATE_BOOLEAN ) ) {
				$params['autoplay'] = '';
			}
		}

		if ( filter_var( $settings['loop'], FILTER_VALIDATE_BOOLEAN ) ) {
			$params['loop'] = '';
		}

		$params['style'] = 'max-width: 100%;';
		$controls        = array( 'playpause', 'progress', 'current', 'duration', 'volume', 'fullscreen' );

		if ( in_array( 'current', $controls ) ) {
			$controls[1] = 'current';
			$controls[2] = 'progress';
		}

		$params['data-controls'] = esc_attr( json_encode( $controls ) );

		return $params;
	}

	/**
	 * Return embedded video parameters
	 *
	 * @return array
	 */
	public function __get_embed_video_params() {

		$settings          = $this->get_settings();
		$params            = array();
		$params_dictionary = array();

		switch ( jet_woo_gallery_video_integration()->get_video_type() ) {
			case 'youtube':
				$params_dictionary = array(
					'autoplay' => 'autoplay',
					'loop'     => 'loop',
				);


				if ( $settings['loop'] ) {
					$video_properties = Embed::get_video_properties( esc_url( $this->__get_video_url() ) );

					$params['playlist'] = $video_properties['video_id'];
				}

				break;
			case 'vimeo':
				$params_dictionary = array(
					'autoplay' => 'autoplay',
					'loop'     => 'loop',
				);

				if ( 'content' === $settings['video_display_in'] ) {
					$params_dictionary['autoplay'] = 'autoplay';
				} else {
					$params_dictionary['autoplay'] = false;
				}

				$params['autopause'] = '0';

				break;
		}

		foreach ( $params_dictionary as $setting_name => $param_name ) {

			$param_value = filter_var( $settings[ $setting_name ], FILTER_VALIDATE_BOOLEAN ) ? '1' : '0';

			$params[ $param_name ] = $param_value;

			if ( 'popup' === $settings['video_display_in'] ) {
				$params['autoplay'] = '0';
			}

		}

		return $params;
	}

	/**
	 * Return video url
	 *
	 * @return string
	 */
	public function __get_video_url() {
		$video_url = '';

		switch ( jet_woo_gallery_video_integration()->get_video_type() ) {
			case 'self_hosted':
				$video_id  = jet_woo_gallery_video_integration()->get_self_hosted_video_id();
				$video_url = wp_get_attachment_url( $video_id );
				break;
			case 'youtube':
				$video_url = jet_woo_gallery_video_integration()->get_youtube_video_url();
				break;
			case 'vimeo':
				$video_url = jet_woo_gallery_video_integration()->get_vimeo_video_url();
				break;
		}

		if ( ! $video_url ) {
			return '';
		}

		return esc_url( $video_url );
	}

}
